<?php
/* --------------------------------------------------------------
 SimpleCustomerService.php 2020-02-14
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Customer;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use InvalidArgumentException;

/**
 * Class SimpleCustomerService
 * @package Gambio\Admin\Customer
 */
class SimpleCustomerService
{
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * SimpleCustomerService constructor.
     *
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }
    
    
    /**
     * Returns the customer name.
     *
     * @param int $customerId
     *
     * @return CustomerName
     */
    public function getName(int $customerId): CustomerName
    {
        $qb = $this->connection->createQueryBuilder();
        
        $result = $qb->select('customers_firstname as first_name, customers_lastname as last_name')
            ->from('customers')
            ->where($this->customerIdEquals($customerId, $qb))
            ->execute()
            ->fetch();
        
        if (!is_array($result)) {
            throw new InvalidArgumentException("Customer with id ({$customerId}) not found.");
        }
        
        return CustomerName::create($result['first_name'], $result['last_name']);
    }
    
    
    /**
     * Checks if customer with given id is an admin.
     *
     * This function evaluates to true if the customer status id equals 0,
     * which represents the admin id in the shop system.
     *
     * @param int $customerId
     *
     * @return bool
     */
    public function isAdmin(int $customerId): bool
    {
        $key    = 'customers_status';
        $qb     = $this->connection->createQueryBuilder();
        $result = $qb->select($key)
            ->from('customers')
            ->where($this->customerIdEquals($customerId, $qb))
            ->execute()
            ->fetch();
        
        if (!is_array($result)) {
            return false;
        }
        
        return $result[$key] === '0';
    }
    
    
    /**
     * Sets prepared statement parameter for where condition of column "customers_id".
     *
     * @param int          $customerId
     * @param QueryBuilder $qb
     *
     * @return string
     */
    private function customerIdEquals(int $customerId, QueryBuilder $qb): string
    {
        return "customers_id = {$qb->createNamedParameter($customerId)}";
    }
}